import os
import json
import requests
import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext
from datetime import datetime

# ==========================
# CONFIGURACIONES
# ==========================
CHANNELS = {
    "tnannastein": ["pickup", "delivery"],
    "magento": ["pickup", "delivery"],
    "MELI": ["pickup", "delivery"],
    "farmaciadigital": ["pickup", "delivery"],
    "farmatouch": ["pickup", "delivery"],
    "RAPPI": ["delivery"],
    "PEYA": ["delivery"],
    "benvida": ["pickup", "delivery"],
    "vtex": ["delivery"]
}

EXTERNAL_FILE = "external_number.txt"
INITIAL_EXTERNAL = 40001400
URL = "https://bmc-foco.dev.napse.global:8444/order/create"
SHIPPING_COST = 15
PROMO_DISCOUNT = 2400

ITEMS = [
    {"itemCode": "1477", "description": "BRAVA DESOD. LUBRICANTE-sachet cajax20 cód.12060", "unitPrice": 5000, "qty": 2},
    {"itemCode": "9018", "description": "BETAVALMAX-60 mg gts.x 15 ml", "unitPrice": 1000, "qty": 1}
]

STORES = {
    "100000001": "Tienda 1",
    "100000002": "Tienda 2",
    "100000003": "Tienda 3",
    "100000004": "Tienda 4"
}


# ==========================
# FUNCIONES
# ==========================
def get_next_external_number():
    if os.path.exists(EXTERNAL_FILE):
        with open(EXTERNAL_FILE, "r") as f:
            num = int(f.read().strip())
    else:
        num = INITIAL_EXTERNAL - 1
    next_num = num + 1
    with open(EXTERNAL_FILE, "w") as f:
        f.write(str(next_num))
    return str(next_num)


def generar_pedido():
    channel = canal_var.get()
    delivery_modes = CHANNELS[channel]

    if len(delivery_modes) == 1:
        delivery = delivery_modes[0]
    else:
        delivery = entrega_var.get()

    storeCode = store_var.get()
    isPaid = pagado_var.get()
    aplicaPromo = promo_var.get()
    aplica_desc_manual = desc_var.get()

    creationDate = datetime.now().strftime("%Y-%m-%d")
    externalNumber = get_next_external_number()

    pedido = {
        "externalNumber": externalNumber,
        "creationDate": creationDate,
        "channelCode": channel,
        "currencyCode": "ARS",
        "manualDiscountAmount": 0,
        "netAmount": sum(i["unitPrice"] * i["qty"] for i in ITEMS),
        "orderTypeCode": "web",
        "isPaid": isPaid,
        "party": {"firstName": "Alejandro", "lastName": "Isola"},
        "detail": []
    }

    for i in ITEMS:
        pedido["detail"].append({
            "itemCode": i["itemCode"],
            "description": i["description"],
            "storeCode": storeCode,
            "locationCode": "DEP1",
            "unitPrice": i["unitPrice"],
            "orderedQty": i["qty"],
            "deliveryOrPickup": delivery
        })

    # Envío
    if delivery == "delivery":
        pedido["detail"].append({
            "itemCode": "ENVIO",
            "description": "Costo de Envio",
            "storeCode": storeCode,
            "locationCode": "DEP1",
            "unitPrice": SHIPPING_COST,
            "orderedQty": 1,
            "deliveryOrPickup": "delivery"
        })
        pedido["netAmount"] += SHIPPING_COST

    # Descuento manual
    if aplica_desc_manual:
        try:
            desc = float(descuento_entry.get())
            if 0 < desc <= pedido["netAmount"]:
                pedido["manualDiscountAmount"] = desc
                pedido["netAmount"] -= desc
        except:
            messagebox.showwarning("Error", "Descuento inválido")

    # Promo
    if aplicaPromo:
        pedido["detail"][0]["priceModifiers"] = [{
            "percent": 20,
            "amount": PROMO_DISCOUNT,
            "reasonText": "38"
        }]
        pedido["netAmount"] -= PROMO_DISCOUNT

    # Pago
    pedido["payments"] = []
    if isPaid:
        pedido["payments"].append({
            "tender": "EL",
            "amount": pedido["netAmount"],
            "currencyCode": "ARS"
        })

    vista_text.delete(1.0, tk.END)
    vista_text.insert(tk.END, json.dumps(pedido, indent=4, ensure_ascii=False))
    return pedido


def enviar_pedido():
    pedido = generar_pedido()
    token = token_entry.get().strip()
    if not token:
        messagebox.showerror("Error", "Debe ingresar el token")
        return
    headers = {"Content-Type": "application/json", "x-access-token": token}
    try:
        resp = requests.post(URL, headers=headers, json=pedido)
        messagebox.showinfo("Resultado", f"Status {resp.status_code}\n{resp.text}")
    except Exception as e:
        messagebox.showerror("Error", str(e))


# ==========================
# CALLBACKS PARA MOSTRAR/OCULTAR DESCUENTO
# ==========================
def actualizar_descuento(*args):
    if promo_var.get():
        # Si aplica promo, ocultar descuento manual
        desc_var.set(False)
        descuento_label.grid_remove()
        descuento_entry.grid_remove()
    elif desc_var.get():
        # Mostrar campo si está marcado descuento manual
        descuento_label.grid(row=4, column=0, sticky="w")
        descuento_entry.grid(row=4, column=1)
    else:
        # Ocultar si no está marcado
        descuento_label.grid_remove()
        descuento_entry.grid_remove()


# ==========================
# GUI
# ==========================
root = tk.Tk()
root.title("Generador de Pedidos")

frame = ttk.Frame(root, padding=10)
frame.pack(fill="both", expand=True)

# Canal
ttk.Label(frame, text="Canal:").grid(row=0, column=0, sticky="w")
canal_var = tk.StringVar(value=list(CHANNELS.keys())[0])
ttk.Combobox(frame, textvariable=canal_var, values=list(CHANNELS.keys())).grid(row=0, column=1)

# Entrega
ttk.Label(frame, text="Entrega:").grid(row=1, column=0, sticky="w")
entrega_var = tk.StringVar(value="pickup")
ttk.Radiobutton(frame, text="Pickup", variable=entrega_var, value="pickup").grid(row=1, column=1, sticky="w")
ttk.Radiobutton(frame, text="Delivery", variable=entrega_var, value="delivery").grid(row=1, column=2, sticky="w")

# Store
ttk.Label(frame, text="Tienda:").grid(row=2, column=0, sticky="w")
store_var = tk.StringVar(value=list(STORES.keys())[0])
ttk.Combobox(frame, textvariable=store_var, values=list(STORES.keys())).grid(row=2, column=1)

# Checkboxes
pagado_var = tk.BooleanVar()
promo_var = tk.BooleanVar()
desc_var = tk.BooleanVar()

ttk.Checkbutton(frame, text="Pagado", variable=pagado_var).grid(row=3, column=0, sticky="w")
ttk.Checkbutton(frame, text="Aplicar Promo", variable=promo_var).grid(row=3, column=1, sticky="w")
ttk.Checkbutton(frame, text="Descuento Manual", variable=desc_var).grid(row=3, column=2, sticky="w")

# Descuento manual (inicialmente oculto)
descuento_label = ttk.Label(frame, text="Importe Descuento:")
descuento_entry = ttk.Entry(frame)

# Token
ttk.Label(frame, text="Token:").grid(row=5, column=0, sticky="w")
token_entry = ttk.Entry(frame, width=40)
token_entry.grid(row=5, column=1, columnspan=2, sticky="we")

# Botones
ttk.Button(frame, text="Vista Previa", command=generar_pedido).grid(row=6, column=0, pady=5)
ttk.Button(frame, text="Enviar Pedido", command=enviar_pedido).grid(row=6, column=1, pady=5)

# Vista previa
vista_text = scrolledtext.ScrolledText(frame, width=80, height=20)
vista_text.grid(row=7, column=0, columnspan=3, pady=10)

# ==========================
# ENLACE DE EVENTOS
# ==========================
promo_var.trace_add("write", actualizar_descuento)
desc_var.trace_add("write", actualizar_descuento)

root.mainloop()
